const Discord = require("discord.js-selfbot-v13");
const { Client, GatewayIntentBits, EmbedBuilder } = require("discord.js");
const fs = require("fs");
require('dotenv').config();


//AC PEPE BERSAMA
const config = {
	userToken: process.env.USER_TOKEN,
	botToken: process.env.BOT_TOKEN,
	channelMapping: {
		"1215350100700831775": "1215332150404980738",
		"1153016190004891678": "1214441302477766707",
		"1258709288994607250": "1214444966865141771",
		"1153033344804733029": "1214160520698134608",
		"1186482371596398593": "1214178504480919562",
		"1156425621661032459": "1214175263034843166",
		"1228962492076785785": "1229040604793798697",
		"1204244802896535633": "1224319055578791976",
		"1241256779564711938": "1241302220222496790",
		"1241975930306428998": "1242706760049885194",
		"1184295348466876437": "1214172807337087036",
		"1426245193648312420": "1426812703930056724",
		// "1154734446147272815": "1214444912519413791",
		"1250988860624736278": "1250993415718699208",
		"1319121612750061608": "1319152587211018240",
	},
	logFile: "message_map.json",
};
class MessageForwarder {
	constructor(config) {
		this.config = config;
		this.userClient = new Discord.Client({
			checkUpdate: false,
			syncStatus: false,
		});
		this.botClient = new Client({
			intents: [GatewayIntentBits.Guilds, GatewayIntentBits.GuildMessages, GatewayIntentBits.MessageContent, GatewayIntentBits.GuildMessageReactions],
		});

		try {
			const data = fs.readFileSync(this.config.logFile, "utf8");
			this.messageMap = new Map(JSON.parse(data));
		} catch (error) {
			console.log("No existing message map found, creating new one");
			this.messageMap = new Map();
		}

		this.pollMap = new Map();
		this.startTime = Date.now();

		// Bind methods
		this.handleUserMessage = this.handleUserMessage.bind(this);
		this.handleBotMessage = this.handleBotMessage.bind(this);
		this.handleUserReady = this.handleUserReady.bind(this);
		this.handleBotReady = this.handleBotReady.bind(this);
	}

	saveMessageMap() {
		const mapArray = Array.from(this.messageMap.entries());
		fs.writeFileSync(this.config.logFile, JSON.stringify(mapArray), "utf8");
	}

	createInitialPollEmbed(poll, author) {
		if (!poll) return null;

		const embed = new EmbedBuilder()
			.setColor(0x2b2d31)
			.setAuthor({
				name: author.globalName || author.username,
				iconURL: author.displayAvatarURL({ dynamic: true }),
			})
			.setTitle(poll.question.text)
			.setFooter({ text: "Poll" })
			.setTimestamp();

		const options = Array.from(poll.answers.values());
		const optionsText = options.map((answer, index) => {
			const letterEmoji = String.fromCodePoint(0x1f1e6 + index);
			return {
				name: `${letterEmoji} Option ${index + 1}`,
				value: `${answer.text}\n${"⬜".repeat(10)} 0%\n0 votes`,
				inline: false,
			};
		});

		embed.addFields(optionsText);
		embed.addFields({
			name: "\u200B",
			value: `0 total votes`,
			inline: false,
		});

		return embed;
	}

	createPollResultEmbed(pollData, author) {
		if (!pollData) return null;

		const percentage = Math.round((pollData.victor_answer_votes / pollData.total_votes) * 100);

		const embed = new EmbedBuilder()
			.setColor(0x2b2d31)
			.setDescription(`${pollData.poll_question_text}\n\n${pollData.victor_answer_text}`)
			.addFields([
				{
					name: pollData.victor_answer_text,
					value: `Winning answer • ${percentage}%`,
					inline: false,
				},
			])
			.setTimestamp();

		return { embed };
	}

	async handlePingCommand(channel) {
		try {
			const latency = Math.round(this.botClient.ws.ping);

			const embed = new EmbedBuilder()
				.setColor(0x2b2d31)
				.setTitle("🤖 Bot Status")
				.setDescription("Informasi status bot saat ini:")
				.addFields([
					{
						name: "📊 Latency",
						value: `${latency}ms`,
						inline: true,
					},
					{
						name: "🤖 Nama Bot",
						value: this.botClient.user.username,
						inline: true,
					},
					{
						name: "🟢 Status Koneksi",
						value: "Terhubung",
						inline: true,
					},
				]);

			await channel.send({ embeds: [embed] });
		} catch (error) {
			console.error("Error handling ping command:", error);
			await channel.send("Error getting bot status.");
		}
	}

	async handleUserMessage(message) {
		try {
			if (message.content.toLowerCase() === "!ping") {
				const targetChannelId = this.config.channelMapping[message.channel.id];
				if (targetChannelId) {
					const targetChannel = await this.botClient.channels.fetch(targetChannelId);
					await this.handlePingCommand(targetChannel);
				}
				return;
			}

			if (message.channel.id in this.config.channelMapping) {
				await this.forwardMessage(message);
			}
		} catch (error) {
			console.error("Error in user message handler:", error);
		}
	}

	async handleBotMessage(message) {
		try {
			if (message.content.toLowerCase() === "!ping") {
				await this.handlePingCommand(message.channel);
				return;
			}
		} catch (error) {
			console.error("Error in bot message handler:", error);
		}
	}

	async forwardMessage(message) {
		try {
			const targetChannelId = this.config.channelMapping[message.channel.id];
			const targetChannel = await this.botClient.channels.fetch(targetChannelId);

			// Prepare components
			const senderName = message.author.globalName || message.author.username;
			const timestamp = `<t:${Math.floor(message.createdTimestamp / 1000)}:F>`;
			const header = `**${senderName}** • ${timestamp}`;
			const tag = message.channel.id !== "1214441302477766707" ? "\n<@&1214160101901082664>" : "";
			const fullHeader = `${header}${tag}\n`;

			// Length calculations
			const MAX_DISCORD_LENGTH = 2000;
			const headerLength = fullHeader.length;
			const tagLength = tag.length;
			const maxFirstPart = MAX_DISCORD_LENGTH - headerLength;
			const maxSecondPart = MAX_DISCORD_LENGTH - tagLength;

			let messageContent = (message.content || "").replace(/<@&1152655670915637389>/g, "").trim();

			// Auto-truncate if content exceeds maximum possible length
			const maxTotalLength = maxFirstPart + maxSecondPart;
			if (messageContent.length > maxTotalLength) {
				messageContent = messageContent.substring(0, maxTotalLength);
			}

			const messageParts = [];
			const options = {
				allowedMentions: { roles: ["1214160101901082664"] },
			};

			// Split message with intelligent paragraph handling
			if (messageContent.length > maxFirstPart) {
				const firstPartEnd = this.findBestSplitIndex(messageContent, maxFirstPart);
				const firstPart = messageContent.slice(0, firstPartEnd).trim();
				const remainingContent = messageContent.slice(firstPartEnd).trim();

				// Split remaining content for second part with tag space
				const secondPartEnd = this.findBestSplitIndex(remainingContent, maxSecondPart - tagLength);
				const secondPart = remainingContent.slice(0, secondPartEnd).trim();

				messageParts.push(firstPart);
				messageParts.push(secondPart);
			} else {
				messageParts.push(messageContent);
			}

			// Handle special message types
			if (message.type === "POLL_RESULT" || message.type === "POLL_END") {
				const pollData = message.poll;
				const resultEmbed = this.createPollResultEmbed(pollData, message.author);
				options.embeds = [resultEmbed.embed];
			} else if (message.poll) {
				const pollEmbed = this.createInitialPollEmbed(message.poll, message.author);
				options.embeds = [pollEmbed];
				this.pollMap.set(message.id, message.poll);
			} else if (message.embeds?.length > 0) {
				options.embeds = message.embeds.map((embed) => new EmbedBuilder(embed.toJSON()));
			}

			// Send messages with proper formatting
			let sentMessageId;
			for (let i = 0; i < messageParts.length; i++) {
				const currentOptions = { ...options };
				const isLastPart = i === messageParts.length - 1;

				// Construct message content
				if (i === 0) {
					currentOptions.content = `${fullHeader}${messageParts[i]}`;
				} else {
					currentOptions.content = `${messageParts[i]}${tag}`;
				}

				// Add attachments to last part
				if (isLastPart && message.attachments?.size > 0) {
					currentOptions.files = Array.from(message.attachments.values()).map((a) => ({
						attachment: a.url,
						name: a.name,
					}));
				}

				// Handle message replies
				if (message.reference && i === 0) {
					const repliedMessageId = message.reference.messageId;
					const targetReplyId = this.messageMap.get(repliedMessageId);
					if (targetReplyId) {
						currentOptions.reply = { messageReference: targetReplyId };
					}
				}

				const sentMessage = await targetChannel.send(currentOptions);

				// Store mapping for replies
				if (i === 0) {
					sentMessageId = sentMessage.id;
					this.messageMap.set(message.id, sentMessageId);
				}

				await new Promise((resolve) => setTimeout(resolve, 500));
			}

			// Handle poll reactions
			if (message.poll && message.type !== "POLL_END") {
				const sentMessage = await targetChannel.messages.fetch(sentMessageId);
				const optionsCount = message.poll.answers.size;
				for (let i = 0; i < optionsCount; i++) {
					const letterEmoji = String.fromCodePoint(0x1f1e6 + i);
					await sentMessage.react(letterEmoji);
				}
			}

			this.saveMessageMap();
			console.log(`✅ Message forwarded: ${message.id} → ${sentMessageId}`);
		} catch (error) {
			console.error("🚨 Error forwarding message:", error);
		}
	}

	findBestSplitIndex(content, maxPos) {
		const splitPriorities = [
			{
				pattern: "\n\n",
				adjustment: 2,
			},
			{
				pattern: "\n――――――――――――――――――――――\n",
				adjustment: 25,
			},
			{
				pattern: ". ",
				adjustment: 2,
			},
			{
				pattern: "\n",
				adjustment: 1,
			},
			{
				pattern: " ",
				adjustment: 1,
			},
		];

		for (const { pattern, adjustment } of splitPriorities) {
			const index = content.lastIndexOf(pattern, maxPos);
			if (index > -1) {
				return index + adjustment;
			}
		}

		// Fallback to safe split
		return Math.min(maxPos, content.length);
	}

	async handleUserReady() {
		console.log("User client ready!");

		// Jika bot client sudah siap, jalankan pemrosesan pesan
		if (this.botClient.readyAt) {
			console.log("User client siap, mulai memproses pesan...");
			await this.processRecentMessages();
		}
	}

	async handleBotReady() {
		console.log("Bot client ready!");

		// Hanya jalankan bagian ini jika user client sudah siap
		if (!this.userClient.readyAt) {
			console.log("Menunggu user client siap sebelum memproses pesan...");
			return;
		}

		await this.processRecentMessages();
	}

	async processRecentMessages() {
		const today = new Date();
		today.setHours(0, 0, 0, 0);
		const twoDaysAgo = new Date(today);
		twoDaysAgo.setDate(today.getDate() - 2);

		for (const [sourceChannelId, targetChannelId] of Object.entries(this.config.channelMapping)) {
			try {
				const sourceChannel = await this.userClient.channels.fetch(sourceChannelId).catch((err) => {
					console.error(`Tidak dapat mengakses channel ${sourceChannelId}: ${err.message}`);
					return null;
				});

				if (!sourceChannel) {
					console.log(`Channel ${sourceChannelId} tidak ditemukan atau tidak dapat diakses`);
					continue;
				}

				// Pastikan channel memiliki method messages
				if (!sourceChannel.messages) {
					console.log(`Channel ${sourceChannelId} tidak memiliki properti messages`);
					continue;
				}

				const messages = await sourceChannel.messages.fetch({ limit: 100 }).catch((err) => {
					console.error(`Gagal mengambil pesan dari channel ${sourceChannelId}: ${err.message}`);
					return null;
				});

				if (!messages) continue;

				const recentMessages = messages.filter((msg) => {
					const messageDate = new Date(msg.createdTimestamp);
					return messageDate >= twoDaysAgo;
				});

				const sortedMessages = Array.from(recentMessages.values()).sort((a, b) => a.createdTimestamp - b.createdTimestamp);

				for (const message of sortedMessages) {
					if (!this.messageMap.has(message.id)) {
						await this.forwardMessage(message);
						// Add delay to prevent rate limiting
						await new Promise((resolve) => setTimeout(resolve, 1000));
					}
				}
			} catch (error) {
				console.error(`Error processing channel ${sourceChannelId}:`, error);
			}
		}
	}

	async start() {
		try {
			// Set up user client event handlers
			this.userClient.on("messageCreate", this.handleUserMessage);
			this.userClient.once("ready", this.handleUserReady);

			// Set up bot client event handlers
			this.botClient.on("messageCreate", this.handleBotMessage);
			this.botClient.once("ready", this.handleBotReady);

			// Login both clients
			await Promise.all([this.userClient.login(this.config.userToken), this.botClient.login(this.config.botToken)]);
		} catch (error) {
			console.error("Error starting the bot:", error);
		}
	}
}

// Create instance and start
const messageForwarder = new MessageForwarder(config);
messageForwarder.start().catch(console.error);
